----------------
title : C++ 레퍼런스 - utility 라이브러리
cat_title :  <utility>
ref_title : <utility>
path : /C++ Reference/utility
publish_date : 2019-09-20
----------------

##@ cpp-ref-start

#@ utility

C++ 의 `<utility>` 라이브러리에는 여러가지 유용한 함수와 클래스들을 제공하고 있습니다. (~~사실 따로 라이브러리 <> 를 부여하기에는 너무 작아서 애매한 애들~~)

### 포함하고 있는 라이브러리

* `<initializer_list>`

### 함수들

* `swap` : 두 객체의 값을 서로 바꿉니다.
* `exchange` : 어떠한 객체의 값을 다른 객체로 바꾼 뒤에, 이전 객체를 리턴합니다.
* `forward` : 인자를 전달합니다.
* `move` : 우측값 레퍼런스를 얻습니다.
* `move_if_noexcept` : 인자의 이동 생성자가 예외를 발생시키지 않는다면 우측값 레퍼런스를 얻습니다.
* `as_const` : 인자의 상수 레퍼런스를 얻습니다.
* `declval` : 인자의 레퍼런스를 얻습니다. 이 때 해당 레퍼런스는 **unevaluated context** (예를 들어서 `decltype` 에 전달시) 에서만 사용되어야만 합니다.
* `make_pair` : `pair` 객체를 만듭니다.
* `operator==, operator!=, operator<, operator<=, operator>, operator>=` : 두 `pair` 의 값을 비교합니다.
* `std::swap(std::pair)` : `swap` 함수의 `std::pair` 를 인자로 받는 템플릿 특수화 버전.
* `std::get(std::pair)` : `pair` 의 인자에 접근한다.

### 클래스

* `pair` : 두 개의 값을 보관하는 클래스
* `integer_sequence` : 컴파일 타임 정수 수열을 만드는 클래스
* `tuple` : 임의의 개수의 서로 다른 클래스의 객체들을 보관. `pair` 의 확장판


### 몇 가지 헬퍼 클래스들

* `piecewise_construct_t, piecewise_construct` : 예상치 못하게 `tuple` 객체를 만드는 것을 막아주는 헬퍼 클래스. 자세한 내용은 해당 클래스 설명 참조
* `in_place, in_place_type, in_place_index, in_place_t, in_place_type_t, in_place_index_t` : 위와 비슷한 역할은 하는데, `std::optional, std::variant, std::any` 에서 사용된다.

### 헤더파일 예시

```cpp-formatted
#include <initializer_list>
namespace std {
// operators:
namespace rel_ops {
template <class T>
bool operator!=(const T&, const T&);
template <class T>
bool operator>(const T&, const T&);
template <class T>
bool operator<=(const T&, const T&);
template <class T>
bool operator>=(const T&, const T&);
}  // namespace rel_ops
// swap:
template <class T>
constexpr void swap(T& a, T& b) noexcept(/*see definition*/);
template <class T, size_t N>
constexpr void swap(T (&a)[N], T (&b)[N]) noexcept(is_nothrow_swappable_v<T>);

// exchange:
template <class T, class U = T>
constexpr T exchange(T& obj, U&& new_val);

// forward/move:
template <class T>
constexpr T&& forward(remove_reference_t<T>& t) noexcept;
template <class T>
constexpr T&& forward(remove_reference_t<T>&& t) noexcept;
template <class T>
constexpr remove_reference_t<T>&& move(T&&) noexcept;
template <class T>
constexpr conditional_t<!is_nothrow_move_constructible_v<T> &&
                          is_copy_constructible_v<T>,
                        const T&, T&&>
move_if_noexcept(T& x) noexcept;

// as_const:
template <class T>
constexpr add_const_t<T>& as_const(T& t) noexcept;
template <class T>
void as_const(const T&&) = delete;

// declval:
template <class T>
add_rvalue_reference_t<T> declval() noexcept;  // as unevaluated operand

// Compile-time integer sequences
template <class T, T...>
struct integer_sequence;
template <size_t... I>
using index_sequence = integer_sequence<size_t, I...>;
template <class T, T N>
using make_integer_sequence = integer_sequence<T, /*see definition*/>;
template <size_t N>
using make_index_sequence = make_integer_sequence<size_t, N>;
template <class... T>
using index_sequence_for = make_index_sequence<sizeof...(T)>;

// pairs:
template <class T1, class T2>
struct pair;

// pair specialized algorithms:
template <class T1, class T2>
constexpr bool operator==(const pair<T1, T2>&, const pair<T1, T2>&);
template <class T1, class T2>
constexpr bool operator<(const pair<T1, T2>&, const pair<T1, T2>&);
template <class T1, class T2>
constexpr bool operator!=(const pair<T1, T2>&, const pair<T1, T2>&);
template <class T1, class T2>
constexpr bool operator>(const pair<T1, T2>&, const pair<T1, T2>&);
template <class T1, class T2>
constexpr bool operator>=(const pair<T1, T2>&, const pair<T1, T2>&);
template <class T1, class T2>
constexpr bool operator<=(const pair<T1, T2>&, const pair<T1, T2>&);
template <class T1, class T2>
constexpr void swap(pair<T1, T2>& x,
                    pair<T1, T2>& y) noexcept(noexcept(x.swap(y)));
template <class T1, class T2>
constexpr /*see definition*/ make_pair(T1&&, T2&&);

// tuple-like access to pair:
template <class T>
class tuple_size;
template <size_t I, class T>
class tuple_element;
template <class T1, class T2>
struct tuple_size<pair<T1, T2>>;
template <class T1, class T2>
struct tuple_element<0, pair<T1, T2>>;
template <class T1, class T2>
struct tuple_element<1, pair<T1, T2>>;
template <size_t I, class T1, class T2>
constexpr tuple_element_t<I, pair<T1, T2>>& get(pair<T1, T2>&) noexcept;
template <size_t I, class T1, class T2>
constexpr tuple_element_t<I, pair<T1, T2>>&& get(pair<T1, T2>&&) noexcept;
template <size_t I, class T1, class T2>
constexpr const tuple_element_t<I, pair<T1, T2>>& get(
  const pair<T1, T2>&) noexcept;
template <size_t I, class T1, class T2>
constexpr const tuple_element_t<I, pair<T1, T2>>&& get(
  const pair<T1, T2>&&) noexcept;
template <class T, class U>
constexpr T& get(pair<T, U>& p) noexcept;
template <class T, class U>
constexpr const T& get(const pair<T, U>& p) noexcept;
template <class T, class U>
constexpr T&& get(pair<T, U>&& p) noexcept;
template <class T, class U>
constexpr const T&& get(const pair<T, U>&& p) noexcept;
template <class T, class U>
constexpr T& get(pair<U, T>& p) noexcept;
template <class T, class U>
constexpr const T& get(const pair<U, T>& p) noexcept;
template <class T, class U>
constexpr T&& get(pair<U, T>&& p) noexcept;
template <class T, class U>
constexpr const T&& get(const pair<U, T>&& p) noexcept;

// pair piecewise construction
struct piecewise_construct_t {};
constexpr piecewise_construct_t piecewise_construct{};
template <class... Types>
class tuple;  // defined in <tuple>

// in-place construction
struct in_place_t {
  explicit in_place_t() = default;
};
inline constexpr in_place_t in_place{};
template <class T>
struct in_place_type_t {
  explicit in_place_type_t() = default;
};
template <class T>
inline constexpr in_place_type_t<T> in_place_type{};
template <size_t I>
struct in_place_index_t {
  explicit in_place_index_t() = default;
};
template <size_t I>
inline constexpr in_place_index_t<I> in_place_index{};
}  // namespace std
```

####  std::integer_sequence 클래스 템플릿

```cpp-formatted
namespace std {
template <class T, T... I>
struct integer_sequence {
  using value_type = T;
  static constexpr size_t size() noexcept { return sizeof...(I); }
};
}  // namespace std
```

#### pair 클래스 템플릿

```cpp-formatted
namespace std {
template <class T1, class T2>
struct pair {
  using first_type = T1;
  using second_type = T2;
  T1 first;
  T2 second;

  pair(const pair&) = default;
  pair(pair&&) = default;

  /*conditionally-explicit*/ constexpr pair();
  /*conditionally-explicit*/ constexpr pair(const T1& x, const T2& y);

  template <class U1, class U2>
  /*conditionally-explicit*/ constexpr pair(U1&& x, U2&& y);
  template <class U1, class U2>
  /*conditionally-explicit*/ constexpr pair(const pair<U1, U2>& p);
  template <class U1, class U2>
  /*conditionally-explicit*/ constexpr pair(pair<U1, U2>&& p);

  template <class... Args1, class... Args2>
  constexpr pair(piecewise_construct_t, tuple<Args1...> arg1,
                 tuple<Args2...> arg2);

  constexpr pair& operator=(const pair& p);

  template <class U1, class U2>
  constexpr pair& operator=(const pair<U1, U2>& p);
  constexpr pair& operator=(pair&& p) noexcept(/*see definition*/);

  template <class U1, class U2>
  constexpr pair& operator=(pair<U1, U2>&& p);
  constexpr void swap(pair& p) noexcept(/*see definition*/);
};

template <class T1, class T2>
pair(T1, T2)->pair<T1, T2>;
}  // namespace std
```